<?php
/**
 *
 * @author Tongle Xu <xutongle@gmail.com> 2013-11-5
 * @copyright Copyright (c) 2003-2103 tintsoft.com
 * @license http://www.tintsoft.com
 * @version $Id$
 */
class request
{
    private static $_requestUri;
    private static $_pathInfo;
    private static $_scriptFile;
    private static $_scriptUrl;
    private static $_hostInfo;
    private static $_baseUrl;
    private static $_cookies;
    private static $_clientIp;
    private static $_preferredAcceptTypes;
    private static $_preferredLanguages;
    private static $_restParams;
    private static $_port;
    private static $_securePort;
    private static $_languages;
    private static $_language;



    /**
     * 路由解析
     *
     * @throws HttpException
     */
    public static function resolve()
    {
        $router = new router();
        $config = pc_base::load_config('urlrouter');
        $router->init($config);
        return $router->parseRequest();
    }

    /**
     * 从GET或POST获取参数
     *
     * @param string $name 参数名称
     * @param string $defaultValue 参数默认值
     * @return string 参数值
     */
    public static function getParam($name, $defaultValue = null)
    {
        return isset ( $_GET [$name] ) ? $_GET [$name] : (isset ( $_POST [$name] ) ? $_POST [$name] : $defaultValue);
    }

    /**
     * 从GET获取参数
     *
     * @param string $name 参数名称
     * @param string $defaultValue 参数默认值
     * @return string 参数值
     */
    public static function getQuery($name, $defaultValue = null)
    {
        return isset ( $_GET [$name] ) ? $_GET [$name] : $defaultValue;
    }

    /**
     * 从POST获取参数
     *
     * @param string $name 参数名称
     * @param string $defaultValue 参数默认值
     * @return string 参数值
     */
    public static function getPost($name, $defaultValue = null)
    {
        return isset ( $_POST [$name] ) ? $_POST [$name] : $defaultValue;
    }

    /**
     * 从Delete获取参数
     *
     * @param string $name 参数名称
     * @param string $defaultValue 参数默认值
     * @return string 参数值
     */
    public static function getDelete($name, $defaultValue = null)
    {
        if ( self::getIsDeleteViaPostRequest () ) return self::getPost ( $name, $defaultValue );
        if ( self::getIsDeleteRequest () ) {
            $restParams = self::getRestParams ();
            return isset ( $restParams [$name] ) ? $restParams [$name] : $defaultValue;
        } else
            return $defaultValue;
    }

    /**
     * 从Put获取参数
     *
     * @param string $name 参数名称
     * @param string $defaultValue 参数默认值
     * @return string 参数值
     */
    public static function getPut($name, $defaultValue = null)
    {
        if ( self::getIsPutViaPostRequest () ) return self::getPost ( $name, $defaultValue );
        if ( self::getIsPutRequest () ) {
            $restParams = self::getRestParams ();
            return isset ( $restParams [$name] ) ? $restParams [$name] : $defaultValue;
        } else
            return $defaultValue;
    }

    /**
     * 返回cookie的值 如果$name=null则返回所有Cookie值
     *
     * @param string $name 获取的变量名,如果该值为null则返回$_COOKIE数组,默认为null
     * @param string $defaultValue 当获取变量失败的时候返回该值,默认该值为null
     * @return mixed
     */
    public static function getCookie($name = null, $defaultValue = null)
    {
        if ( $name === null ) return $_COOKIE;
        return (isset ( $_COOKIE [$name] )) ? $_COOKIE [$name] : $defaultValue;
    }

    /**
     * 返回请求参数
     *
     * @return multitype:
     */
    public static function getRestParams()
    {
        if ( self::$_restParams === null ) {
            $result = array ();
            if ( function_exists ( 'mb_parse_str' ) )
                mb_parse_str ( self::getRawBody (), $result );
            else parse_str ( self::getRawBody (), $result );
            self::$_restParams = $result;
        }
        return self::$_restParams;
    }

    /**
     * 返回原始HTTP请求体
     *
     * @return string 原始HTTP请求体
     */
    public static function getRawBody()
    {
        static $rawBody;
        if ( $rawBody === null ) $rawBody = file_get_contents ( 'php://input' );
        return $rawBody;
    }

    /**
     * 返回当前请求的URL
     *
     * @return string
     */
    public static function getUrl()
    {
        return self::getRequestUri ();
    }

    /**
     * 获得主机信息，包含协议信息，主机名，访问端口信息 <pre>Example: 请求:
     * http://www.tintsoft.com/example/index.php?a=test 返回：
     * http://www.tintsoft.com </pre>
     *
     * @param string $schema 模式(http、https)如果空,使用的模式将使用当前的请求
     * @return string 主机信息
     */
    public static function getHostInfo($schema = '')
    {
        if ( self::$_hostInfo === null ) {
            if ( $secure = self::getIsSecureConnection () )
                $http = 'https';
            else $http = 'http';
            if ( isset ( $_SERVER ['HTTP_HOST'] ) )
                self::$_hostInfo = $http . '://' . $_SERVER ['HTTP_HOST'];
            else {
                self::$_hostInfo = $http . '://' . $_SERVER ['SERVER_NAME'];
                $port = $secure ? self::getSecurePort () : self::getPort ();
                if ( ($port !== 80 && ! $secure) || ($port !== 443 && $secure) ) self::$_hostInfo .= ':' . $port;
            }
        }
        if ( $schema !== '' ) {
            $secure = self::getIsSecureConnection ();
            if ( $secure && $schema === 'https' || ! $secure && $schema === 'http' ) return self::$_hostInfo;

            $port = $schema === 'https' ? self::getSecurePort () : self::getPort ();
            if ( $port !== 80 && $schema === 'http' || $port !== 443 && $schema === 'https' )
                $port = ':' . $port;
            else $port = '';

            $pos = strpos ( self::$_hostInfo, ':' );
            return $schema . substr ( self::$_hostInfo, $pos, strcspn ( self::$_hostInfo, ':', $pos + 1 ) + 1 ) . $port;
        } else
            return self::$_hostInfo;
    }

    /**
     * 获取请求链接协议 如果是安全链接请求则返回https否则返回http
     *
     * @return string
     */
    public static function getScheme()
    {
        return (self::getIsSecureConnection ()) ? 'https' : 'http';
    }

    /**
     * 设置主机信息
     *
     * @param unknown $value
     */
    public static function setHostInfo($value)
    {
        self::$_hostInfo = rtrim ( $value, '/' );
    }

    /**
     * 获取基础URL 这里是去除了脚本文件以及访问参数信息的URL地址信息: <pre>Example: 请求:
     * http://www.tintsoft.com/example/index.php?a=test 1]如果: $absolute = false：
     * 返回： example 2]如果: $absolute = true: 返回： http://www.tintsoft.com/example
     * </pre>
     *
     * @param string $absolute
     * @return string 基础URL
     */
    public static function getBaseUrl($absolute = false)
    {
        if ( self::$_baseUrl === null ) self::$_baseUrl = rtrim ( dirname ( self::getScriptUrl () ), '\\/' );
        return $absolute ? self::getHostInfo () . self::$_baseUrl : self::$_baseUrl;
    }

    /**
     * 设置基础URL
     *
     * @param string $value 基础URL
     */
    public static function setBaseUrl($value)
    {
        self::$_baseUrl = $value;
    }

    /**
     * 返回当前执行脚本的绝对路径 <pre>Example: 请求:
     * http://www.tintsoft.com/example/index.php?a=test 返回: /example/index.php
     * </pre>
     *
     * @throws Exception
     * @return string 当前执行脚本的绝对路径
     */
    public static function getScriptUrl()
    {
        if ( self::$_scriptUrl === null ) {
            $scriptName = basename ( $_SERVER ['SCRIPT_FILENAME'] );
            if ( basename ( $_SERVER ['SCRIPT_NAME'] ) === $scriptName )
                self::$_scriptUrl = $_SERVER ['SCRIPT_NAME'];
            elseif ( basename ( $_SERVER ['PHP_SELF'] ) === $scriptName )
                self::$_scriptUrl = $_SERVER ['PHP_SELF'];
            elseif ( isset ( $_SERVER ['ORIG_SCRIPT_NAME'] ) && basename ( $_SERVER ['ORIG_SCRIPT_NAME'] ) === $scriptName )
                self::$_scriptUrl = $_SERVER ['ORIG_SCRIPT_NAME'];
            elseif ( ($pos = strpos ( $_SERVER ['PHP_SELF'], '/' . $scriptName )) !== false )
                self::$_scriptUrl = substr ( $_SERVER ['SCRIPT_NAME'], 0, $pos ) . '/' . $scriptName;
            elseif ( isset ( $_SERVER ['DOCUMENT_ROOT'] ) && strpos ( $_SERVER ['SCRIPT_FILENAME'], $_SERVER ['DOCUMENT_ROOT'] ) === 0 )
                self::$_scriptUrl = str_replace ( '\\', '/', str_replace ( $_SERVER ['DOCUMENT_ROOT'], '', $_SERVER ['SCRIPT_FILENAME'] ) );
            else throw new \Exception ( 'HttpRequest is unable to determine the entry script URL.' );
        }
        return self::$_scriptUrl;
    }

    /**
     * 设置当前执行脚本的绝对路径
     *
     * @param string $value 当前执行脚本的绝对路径
     */
    public function setScriptUrl($value)
    {
        self::$_scriptUrl = '/' . trim ( $value, '/' );
    }

    /**
     * 返回当前请求的URL路径信息。 返回包含由客户端提供的、跟在真实脚本名称之后并且在查询语句（query string）之前的路径信息
     * <pre>Example: 请求:
     * http://www.tintsoft.com/example/index.php/site/main?a=test 返回: site/main
     * </pre>
     *
     * @throws Exception
     * @return string 当前请求的URL的路径信息
     */
    public static function getPathInfo()
    {
        if ( self::$_pathInfo === null ) {
            $pathInfo = self::getRequestUri ();
            if ( ($pos = strpos ( $pathInfo, '?' )) !== false ) $pathInfo = substr ( $pathInfo, 0, $pos );
            $pathInfo = self::decodePathInfo ( $pathInfo );
            $scriptUrl = self::getScriptUrl ();
            $baseUrl = self::getBaseUrl ();
            if ( strpos ( $pathInfo, $scriptUrl ) === 0 )
                $pathInfo = substr ( $pathInfo, strlen ( $scriptUrl ) );
            elseif ( $baseUrl === '' || strpos ( $pathInfo, $baseUrl ) === 0 )
                $pathInfo = substr ( $pathInfo, strlen ( $baseUrl ) );
            elseif ( strpos ( $_SERVER ['PHP_SELF'], $scriptUrl ) === 0 )
                $pathInfo = substr ( $_SERVER ['PHP_SELF'], strlen ( $scriptUrl ) );
            else throw new \Exception ( 'Request is unable to determine the path info of the request.' );
            self::$_pathInfo = trim ( $pathInfo, '/' );
        }
        return self::$_pathInfo;
    }

    /**
     * 解码路径信息
     *
     * @param string $pathInfo
     * @return string 路径信息
     */
    protected static function decodePathInfo($pathInfo)
    {
        $pathInfo = urldecode ( $pathInfo );
        if ( preg_match ( '%^(?:
		   [\x09\x0A\x0D\x20-\x7E]            # ASCII
		 | [\xC2-\xDF][\x80-\xBF]             # non-overlong 2-byte
		 | \xE0[\xA0-\xBF][\x80-\xBF]         # excluding overlongs
		 | [\xE1-\xEC\xEE\xEF][\x80-\xBF]{2}  # straight 3-byte
		 | \xED[\x80-\x9F][\x80-\xBF]         # excluding surrogates
		 | \xF0[\x90-\xBF][\x80-\xBF]{2}      # planes 1-3
		 | [\xF1-\xF3][\x80-\xBF]{3}          # planes 4-15
		 | \xF4[\x80-\x8F][\x80-\xBF]{2}      # plane 16
		)*$%xs', $pathInfo ) ) {
            return $pathInfo;
        } else {
            return utf8_encode ( $pathInfo );
        }
    }

    /**
     * 返回请求的资源标识符 <pre>这里的uri是去除协议名、主机名的 Example: 请求：
     * http://www.tintsoft.com/example/index.php?a=test 则返回:
     * /example/index.php?a=test </pre>
     *
     * @throws Exception
     * @return string 返回请求的资源标识符
     */
    public static function getRequestUri()
    {
        if ( self::$_requestUri === null ) {
            if ( isset ( $_SERVER ['HTTP_X_REWRITE_URL'] ) ) // IIS
                self::$_requestUri = $_SERVER ['HTTP_X_REWRITE_URL'];
            elseif ( isset ( $_SERVER ['REQUEST_URI'] ) ) {
                self::$_requestUri = $_SERVER ['REQUEST_URI'];
                if ( ! empty ( $_SERVER ['HTTP_HOST'] ) ) {
                    if ( strpos ( self::$_requestUri, $_SERVER ['HTTP_HOST'] ) !== false ) self::$_requestUri = preg_replace ( '/^\w+:\/\/[^\/]+/', '', self::$_requestUri );
                } else
                    self::$_requestUri = preg_replace ( '/^(http|https):\/\/[^\/]+/i', '', self::$_requestUri );
            } elseif ( isset ( $_SERVER ['ORIG_PATH_INFO'] ) ) {
                self::$_requestUri = $_SERVER ['ORIG_PATH_INFO'];
                if ( ! empty ( $_SERVER ['QUERY_STRING'] ) ) self::$_requestUri .= '?' . $_SERVER ['QUERY_STRING'];
            } else
                throw new \Exception ( 'HttpRequest is unable to determine the request URI.' );
        }
        return self::$_requestUri;
    }

    /**
     * 返回查询字符串
     *
     * @return string 查询字符串
     */
    public static function getQueryString()
    {
        return isset ( $_SERVER ['QUERY_STRING'] ) ? $_SERVER ['QUERY_STRING'] : '';
    }

    /**
     * 请求是否使用的是HTTPS安全链接 如果是安全请求则返回true否则返回false
     *
     * @return boolean 是否使用的是HTTPS安全链接
     */
    public static function getIsSecureConnection()
    {
        return isset ( $_SERVER ['HTTPS'] ) && ($_SERVER ['HTTPS'] == 'on' || $_SERVER ['HTTPS'] == 1) || isset ( $_SERVER ['HTTP_X_FORWARDED_PROTO'] ) && $_SERVER ['HTTP_X_FORWARDED_PROTO'] == 'https';
    }

    /**
     * 获得请求类型
     *
     * @return string 请求类型
     */
    public static function getRequestType()
    {
        if ( isset ( $_POST ['_method'] ) ) return strtoupper ( $_POST ['_method'] );
        return strtoupper ( isset ( $_SERVER ['REQUEST_METHOD'] ) ? $_SERVER ['REQUEST_METHOD'] : 'GET' );
    }

    /**
     * 返回是否是GET请求
     *
     * @return boolean 是否是GET请求
     */
    public static function getIsGetRequest()
    {
        return isset ( $_SERVER ['REQUEST_METHOD'] ) && ! strcasecmp ( $_SERVER ['REQUEST_METHOD'], 'Get' );
    }

    /**
     * 返回是否是POST请求
     *
     * @return boolean 是否是POST请求
     */
    public static function getIsPostRequest()
    {
        return isset ( $_SERVER ['REQUEST_METHOD'] ) && ! strcasecmp ( $_SERVER ['REQUEST_METHOD'], 'POST' );
    }

    /**
     * 返回是否是Delete请求
     *
     * @return boolean 是否是Delete请求
     */
    public static function getIsDeleteRequest()
    {
        return (isset ( $_SERVER ['REQUEST_METHOD'] ) && ! strcasecmp ( $_SERVER ['REQUEST_METHOD'], 'DELETE' )) || self::getIsDeleteViaPostRequest ();
    }

    /**
     * 返回是否是Delete请求,通过POST
     *
     * @return boolean 是否是Delete请求
     */
    protected static function getIsDeleteViaPostRequest()
    {
        return isset ( $_POST ['_method'] ) && ! strcasecmp ( $_POST ['_method'], 'DELETE' );
    }

    /**
     * 返回是否是PUT请求
     *
     * @return boolean 是否是PUT请求
     */
    public static function getIsPutRequest()
    {
        return (isset ( $_SERVER ['REQUEST_METHOD'] ) && ! strcasecmp ( $_SERVER ['REQUEST_METHOD'], 'PUT' )) || self::getIsPutViaPostRequest ();
    }

    /**
     * 返回是否是PUT请求，通过POST
     *
     * @return boolean 是否是PUT请求
     */
    protected static function getIsPutViaPostRequest()
    {
        return isset ( $_POST ['_method'] ) && ! strcasecmp ( $_POST ['_method'], 'PUT' );
    }

    /**
     * 返回是否这是一个AJAX(XMLHttpRequest)请求
     *
     * @return boolean 是否这是一个AJAX(XMLHttpRequest)请求
     */
    public static function getIsAjaxRequest()
    {
        return isset ( $_SERVER ['HTTP_X_REQUESTED_WITH'] ) && $_SERVER ['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest';
    }

    /**
     * 返回是否是Adobe Flash或Adobe Flex的请求
     *
     * @return boolean 是否是Adobe Flash或Adobe Flex的请求
     */
    public static function getIsFlashRequest()
    {
        return isset ( $_SERVER ['HTTP_USER_AGENT'] ) && (stripos ( $_SERVER ['HTTP_USER_AGENT'], 'Shockwave' ) !== false || stripos ( $_SERVER ['HTTP_USER_AGENT'], 'Flash' ) !== false);
    }

    /**
     * 返回当前运行脚本所在的服务器的主机名。 如果脚本运行于虚拟主机中 该名称是由那个虚拟主机所设置的值决定
     *
     * @return string 主机名
     */
    public static function getServerName()
    {
        return $_SERVER ['SERVER_NAME'];
    }

    /**
     * 返回服务端口号 https链接的默认端口号为443 http链接的默认端口号为80
     *
     * @return string 端口号
     */
    public static function getServerPort()
    {
        return $_SERVER ['SERVER_PORT'];
    }

    /**
     * 返回浏览器发送Referer请求头 可以让服务器了解和追踪发出本次请求的起源URL地址
     *
     * @return string 来源URL地址
     */
    public static function getUrlReferrer()
    {
        return isset ( $_SERVER ['HTTP_REFERER'] ) ? $_SERVER ['HTTP_REFERER'] : null;
    }

    /**
     * 返回User-Agent头字段用于指定浏览器或者其他客户端程序的类型和名字
     * 如果客户机是一种无线手持终端，就返回一个WML文件；如果发现客户端是一种普通浏览器， 则返回通常的HTML文件
     *
     * @return string User-Agent字段内容
     */
    public static function getUserAgent()
    {
        return isset ( $_SERVER ['HTTP_USER_AGENT'] ) ? $_SERVER ['HTTP_USER_AGENT'] : null;
    }

    /**
     * 返回用户IP地址 如果获取请求IP失败,则返回0.0.0.0
     *
     * @return string 用户IP地址
     */
    public static function getUserIp()
    {
        if ( ! self::$_clientIp ) {
            if ( isset ( $_SERVER ['HTTP_CLIENT_IP'] ) ) {
                self::$_clientIp = $_SERVER ['HTTP_CLIENT_IP'];
            } elseif ( isset ( $_SERVER ['HTTP_X_FORWARDED_FOR'] ) ) {
                $ip = strtok ( $_SERVER ['HTTP_X_FORWARDED_FOR'], ',' );
                do {
                    $ip = ip2long ( $ip );
                    if ( ! (($ip == 0) || ($ip == 0xFFFFFFFF) || ($ip == 0x7F000001) || (($ip >= 0x0A000000) && ($ip <= 0x0AFFFFFF)) || (($ip >= 0xC0A8FFFF) && ($ip <= 0xC0A80000)) || (($ip >= 0xAC1FFFFF) && ($ip <= 0xAC100000))) ) {
                        self::$_clientIp = long2ip ( $ip );
                        return;
                    }
                }
                while ( ($ip = strtok ( ',' )) );
            } elseif ( isset ( $_SERVER ['HTTP_PROXY_USER'] ) ) {
                self::$_clientIp = $_SERVER ['HTTP_PROXY_USER'];
            } elseif ( isset ( $_SERVER ['REMOTE_ADDR'] ) ) {
                self::$_clientIp = $_SERVER ['REMOTE_ADDR'];
            } else {
                self::$_clientIp = "0.0.0.0";
            }
        }
        return self::$_clientIp;
    }

    /**
     * 获取客户端浏览器信息
     *
     * @access public
     * @return string
     */
    public static function getUserBrowser()
    {
        $userAgent = self::getUserAgent ();
        if ( strpos ( $userAgent, 'MSIE 10.0' ) ) {
            return 'IE10';
        } elseif ( strpos ( $userAgent, 'MSIE 9.0' ) ) {
            return 'IE9';
        } else if ( strpos ( $userAgent, 'MSIE 8.0' ) ) {
            return 'IE8';
        } else if ( strpos ( $userAgent, 'MSIE 7.0' ) ) {
            return 'IE7';
        } else if ( strpos ( $userAgent, 'MSIE 6.0' ) ) {
            return 'IE6';
        } else if ( strpos ( $userAgent, 'Firefox' ) ) {
            return 'Firfox';
        } else if ( strpos ( $userAgent, 'Chrome' ) ) {
            return 'Chrome';
        } else if ( strpos ( $userAgent, 'Opera' ) ) {
            return 'Opera';
        } else if ( strpos ( $userAgent, 'Safari' ) ) {
            return 'Safari';
        } else if ( strpos ( $userAgent, 'Maxthon' ) ) {
            return 'Maxthon';
        } else if ( strpos ( $userAgent, 'MicroMessenger' ) ) {
            return 'Weixin';
        } else if ( strpos ( $userAgent, 'UCWEB' ) ) {
            return 'UcWeb';
        } else if ( strpos ( $userAgent, 'Elinks' ) ) {
            return 'Elinks';
        } else if ( strpos ( $userAgent, 'OmniWeb' ) ) {
            return 'OmniWeb';
        } else if ( strpos ( $userAgent, 'Links' ) ) {
            return 'Links';
        } else if ( strpos ( $userAgent, 'Lynx' ) ) {
            return 'Lynx';
        } else if ( strpos ( $userAgent, 'Arora' ) ) {
            return 'Arora';
        } else if ( strpos ( $userAgent, 'Epiphany' ) ) {
            return 'Epiphany';
        } else if ( strpos ( $userAgent, 'Konqueror' ) ) {
            return 'Konqueror';
        } else if ( strpos ( $userAgent, 'EudoraWeb' ) ) {
            return 'EudoraWeb';
        } else if ( strpos ( $userAgent, 'Minimo' ) ) {
            return 'Minimo';
        } else if ( strpos ( $userAgent, 'NetFront' ) ) {
            return 'NetFront';
        } else if ( strpos ( $userAgent, 'POLARIS' ) ) {
            return 'Polaris';
        } else if ( strpos ( $userAgent, 'BlackBerry' ) ) {
            return 'BlackBerry';
        } else if ( strpos ( $userAgent, 'Nokia' ) ) {
            return 'Nokia';
        } else {
            return 'Others';
        }
    }

    /**
     * 获取客户端操作系统信息
     *
     * @return string
     */
    public static function getUserOs()
    {
        $userAgent = self::getUserAgent ();
        if ( strpos ( $userAgent, 'Windows NT 6.2' ) ) {
            return 'Windows 8';
        } elseif ( strpos ( $userAgent, 'Windows NT 6.1' ) ) {
            return 'Windows 7';
        } else if ( strpos ( $userAgent, 'Windows NT 6.0' ) ) {
            return 'Windows Vista';
        } else if ( strpos ( $userAgent, 'Windows NT 5.2' ) ) {
            return 'Windows 2003';
        } else if ( strpos ( $userAgent, 'Windows NT 5.1' ) ) {
            return 'Windows XP';
        } else if ( strpos ( $userAgent, 'Windows NT 5.0' ) ) {
            return 'Windows 2000';
        } else if ( strpos ( $userAgent, 'Windows ME' ) ) {
            return 'Windows ME';
        } else if ( strpos ( $userAgent, 'PPC Mac OS X' ) ) {
            return 'OS X PPC';
        } else if ( strpos ( $userAgent, 'Intel Mac OS X' ) ) {
            return 'OS X Intel';
        } else if ( strpos ( $userAgent, 'Win98' ) ) {
            return 'Windows 98';
        } else if ( strpos ( $userAgent, 'Win95' ) ) {
            return 'Windows 95';
        } else if ( strpos ( $userAgent, 'WinNT4.0' ) ) {
            return 'Windows NT4.0';
        } else if ( strpos ( $userAgent, 'Mac OS X Mach-O' ) ) {
            return 'OS X Mach';
        } else if ( strpos ( $userAgent, 'Ubuntu' ) ) {
            return 'Ubuntu';
        } else if ( strpos ( $userAgent, 'Debian' ) ) {
            return 'Debian';
        } else if ( strpos ( $userAgent, 'AppleWebKit' ) ) {
            return 'WebKit';
        } else if ( strpos ( $userAgent, 'Mint/8' ) ) {
            return 'Mint 8';
        } else if ( strpos ( $userAgent, 'Minefield' ) ) {
            return 'Minefield Alpha';
        } else if ( strpos ( $userAgent, 'gentoo' ) ) {
            return 'Gentoo';
        } else if ( strpos ( $userAgent, 'Kubuntu' ) ) {
            return 'Kubuntu';
        } else if ( strpos ( $userAgent, 'Slackware/13.0' ) ) {
            return 'Slackware 13';
        } else if ( strpos ( $userAgent, 'Fedora' ) ) {
            return 'Fedora';
        } else if ( strpos ( $userAgent, 'FreeBSD' ) ) {
            return 'FreeBSD';
        } else if ( strpos ( $userAgent, 'SunOS' ) ) {
            return 'SunOS';
        } else if ( strpos ( $userAgent, 'OpenBSD' ) ) {
            return 'OpenBSD';
        } else if ( strpos ( $userAgent, 'NetBSD' ) ) {
            return 'NetBSD';
        } else if ( strpos ( $userAgent, 'DragonFly' ) ) {
            return 'DragonFly';
        } else if ( strpos ( $userAgent, 'IRIX' ) ) {
            return 'IRIX';
        } else if ( strpos ( $userAgent, 'Windows CE' ) ) {
            return 'Windows CE';
        } else if ( strpos ( $userAgent, 'PalmOS' ) ) {
            return 'PalmOS';
        } else if ( strpos ( $userAgent, 'Linux' ) ) {
            return 'Linux';
        } else if ( strpos ( $userAgent, 'DragonFly' ) ) {
            return 'DragonFly';
        } else if ( strpos ( $userAgent, 'Android' ) ) {
            return 'Android';
        } else if ( strpos ( $userAgent, 'Mac OS X' ) ) {
            return 'Mac OS X';
        } else if ( strpos ( $userAgent, 'iPhone' ) ) {
            return 'iPhone OS';
        } else if ( strpos ( $userAgent, 'Symbian OS' ) ) {
            return 'Symbian';
        } else if ( strpos ( $userAgent, 'Symbian OS' ) ) {
            return 'Symbian';
        } else if ( strpos ( $userAgent, 'SymbianOS' ) ) {
            return 'SymbianOS';
        } else if ( strpos ( $userAgent, 'webOS' ) ) {
            return 'webOS';
        } else if ( strpos ( $userAgent, 'PalmSource' ) ) {
            return 'PalmSource';
        } else {
            return 'Others';
        }
    }

    /**
     * 返回用户主机名称,null如果它不能确定
     *
     * @return string 用户主机名称
     */
    public static function getUserHost()
    {
        return isset ( $_SERVER ['REMOTE_HOST'] ) ? $_SERVER ['REMOTE_HOST'] : null;
    }

    /**
     * 获取用户语言
     *
     * @return Ambigous <NULL, string, multitype:>
     */
    public static function getUserLanguages()
    {
        if ( self::$_languages === null ) {
            if ( ! isset ( $_SERVER ['HTTP_ACCEPT_LANGUAGE'] ) )
                self::$_languages [0] = 'en';
            else {
                self::$_languages = array ();
                foreach ( explode ( ',', $_SERVER ['HTTP_ACCEPT_LANGUAGE'] ) as $language ) {
                    $array = explode ( ';q=', trim ( $language ) );
                    self::$_languages [trim ( $array [0] )] = isset ( $array [1] ) ? ( float ) $array [1] : 1.0;
                }
                arsort ( self::$_languages );
                self::$_languages = array_keys ( self::$_languages );
                if ( empty ( self::$_languages ) ) self::$_languages [0] = 'en-us';
            }
        }
        return self::$_languages;
    }

    /**
     * Returns the most preferred language by the client user.
     *
     * @return string the most preferred language by the client user, defaults
     *         to English.
     */
    public static function getPreferredLanguage()
    {
        if ( self::$_language === null ) {
            $langs = self::getUserLanguages ();
            if ( empty ( $langs [0] ) )
                self::$_language = 'en-us';
            else self::$_language = $langs [0];
        }
        return self::$_language;
    }

    /**
     * 返回当前脚本文件的路径
     *
     * @return string 脚本文件路径
     */
    public static function getScriptFile()
    {
        if ( self::$_scriptFile !== null )
            return self::$_scriptFile;
        else return self::$_scriptFile = realpath ( $_SERVER ['SCRIPT_FILENAME'] );
    }

    /**
     * 返回用户浏览器的性能
     *
     * @param string $userAgent
     * @return mixed 用户浏览器的性能
     */
    public static function getBrowser($userAgent = null)
    {
        return get_browser ( $userAgent, true );
    }

    /**
     * 返回用户的浏览器接受类型
     *
     * @return string 用户浏览器接受类型
     */
    public static function getAcceptTypes()
    {
        return isset ( $_SERVER ['HTTP_ACCEPT'] ) ? $_SERVER ['HTTP_ACCEPT'] : null;
    }

    /**
     * 返回端口(非HTTPS)
     *
     * @return number 端口
     */
    public static function getPort()
    {
        if ( self::$_port === null ) self::$_port = ! self::getIsSecureConnection () && isset ( $_SERVER ['SERVER_PORT'] ) ? ( int ) $_SERVER ['SERVER_PORT'] : 80;
        return self::$_port;
    }

    /**
     * 设置端口(非HTTPS)
     *
     * @param number $value 端口
     */
    public static function setPort($value)
    {
        self::$_port = ( int ) $value;
        self::$_hostInfo = null;
    }

    /**
     * 返回端口(HTTPS)
     *
     * @return number 端口
     */
    public static function getSecurePort()
    {
        if ( self::$_securePort === null ) self::$_securePort = self::getIsSecureConnection () && isset ( $_SERVER ['SERVER_PORT'] ) ? ( int ) $_SERVER ['SERVER_PORT'] : 443;
        return self::$_securePort;
    }

    /**
     * 设置端口(HTTPS)
     *
     * @param number $value 端口
     */
    public static function setSecurePort($value)
    {
        self::$_securePort = ( int ) $value;
        self::$_hostInfo = null;
    }

    /**
     * 解析一个HTTP Accept标头,返回一个数组映射 Example:
     * <code>'application/xhtml+xml;q=0.9;level=1'</code> <pre> array( 'type' =>
     * 'application', 'subType' => 'xhtml', 'baseType' => 'xml', 'params' =>
     * array( 'q' => 0.9, 'level' => '1', ), ) </pre>
     *
     * @param string $header Accept标头
     * @return array 数组映射
     */
    public static function parseAcceptHeader($header)
    {
        $matches = array ();
        $accepts = array ();
        // get individual entries with their type, subtype, basetype and params
        preg_match_all ( '/(?:\G\s?,\s?|^)(\w+|\*)\/(\w+|\*)(?:\+(\w+))?|(?<!^)\G(?:\s?;\s?(\w+)=([\w\.]+))/', $header, $matches );
        // the regexp should (in theory) always return an array of 6 arrays
        if ( count ( $matches ) === 6 ) {
            $i = 0;
            $itemLen = count ( $matches [1] );
            while ( $i < $itemLen ) {
                // fill out a content type
                $accept = array ('type' => $matches [1] [$i],'subType' => $matches [2] [$i],'baseType' => null,'params' => array () );
                // fill in the base type if it exists
                if ( $matches [3] [$i] !== null && $matches [3] [$i] !== '' ) $accept ['baseType'] = $matches [3] [$i];
                // continue looping while there is no new content type, to fill
                // in all accompanying params
                for ( $i ++; $i < $itemLen; $i ++ ) {
                    // if the next content type is null, then the item is a
                    // param for the current content type
                    if ( $matches [1] [$i] === null || $matches [1] [$i] === '' ) {
                        // if this is the quality param, convert it to a double
                        if ( $matches [4] [$i] === 'q' ) {
                            // sanity check on q value
                            $q = ( double ) $matches [5] [$i];
                            if ( $q > 1 )
                                $q = ( double ) 1;
                            elseif ( $q < 0 )
                                $q = ( double ) 0;
                            $accept ['params'] [$matches [4] [$i]] = $q;
                        } else
                            $accept ['params'] [$matches [4] [$i]] = $matches [5] [$i];
                    } else
                        break;
                }
                // q defaults to 1 if not explicitly given
                if ( ! isset ( $accept ['params'] ['q'] ) ) $accept ['params'] ['q'] = ( double ) 1;
                $accepts [] = $accept;
            }
        }
        return $accepts;
    }
    public static function compareAcceptTypes($a, $b)
    {
        // check for equal quality first
        if ( $a ['params'] ['q'] === $b ['params'] ['q'] )
            if ( ! ($a ['type'] === '*' xor $b ['type'] === '*') )
                if ( ! ($a ['subType'] === '*' xor $b ['subType'] === '*') )
                    // finally, higher number of parameters counts as greater
                    // precedence
                    if ( count ( $a ['params'] ) === count ( $b ['params'] ) )
                        return 0;
                    else return count ( $a ['params'] ) < count ( $b ['params'] ) ? 1 : - 1;
                    // more specific takes precedence - whichever one doesn't
                    // have a * subType
                else return $a ['subType'] === '*' ? 1 : - 1;
                // more specific takes precedence - whichever one doesn't have a
                // * type
            else return $a ['type'] === '*' ? 1 : - 1;
        else return ($a ['params'] ['q'] < $b ['params'] ['q']) ? 1 : - 1;
    }
    public static function getPreferredAcceptTypes()
    {
        if ( self::$_preferredAcceptTypes === null ) {
            $accepts = self::parseAcceptHeader ( self::getAcceptTypes () );
            print_r ( $accepts );
            usort ( $accepts, array (get_class ( $this ),'compareAcceptTypes' ) );
            self::$_preferredAcceptTypes = $accepts;
        }
        return self::$_preferredAcceptTypes;
    }

    /**
     * 返回用户优先接受的MIME类型。
     *
     * @return array
     */
    public static function getPreferredAcceptType()
    {
        $preferredAcceptTypes = self::getPreferredAcceptTypes ();
        return empty ( $preferredAcceptTypes ) ? false : $preferredAcceptTypes [0];
    }

    /**
     * 返回用户可接受的语言数组
     *
     * @return array 语言数组
     */
    public static function getPreferredLanguages()
    {
        if ( self::$_preferredLanguages === null ) {
            $sortedLanguages = array ();
            if ( isset ( $_SERVER ['HTTP_ACCEPT_LANGUAGE'] ) && $n = preg_match_all ( '/([\w\-_]+)(?:\s*;\s*q\s*=\s*(\d*\.?\d*))?/', $_SERVER ['HTTP_ACCEPT_LANGUAGE'], $matches ) ) {
                $languages = array ();
                for ( $i = 0; $i < $n; ++ $i ) {
                    $q = $matches [2] [$i];
                    if ( $q === '' ) $q = 1;
                    if ( $q ) $languages [] = array (( float ) $q,$matches [1] [$i] );
                }
                usort ( $languages, create_function ( '$a,$b', 'if($a[0]==$b[0]) {return 0;} return ($a[0]<$b[0]) ? 1 : -1;' ) );
                foreach ( $languages as $language )
                    $sortedLanguages [] = $language [1];
            }
            self::$_preferredLanguages = $sortedLanguages;
        }
        return self::$_preferredLanguages;
    }
}
class router
{
    /**
     * 是否启用URL美化
     *
     * @var unknown
     */
    public $enablePrettyUrl = true;

    /**
     * 路由规则
     *
     * @var unknown
     */
    public $rules = array ();

    /**
     * URL美化时使用的后缀
     *
     * @var unknown
     */
    public $suffix;

    /**
     * 显示脚本名称
     *
     * @var unknown
     */
    public $showScriptName = true;

    /**
     * 非URL美化模式的参数名称
     *
     * @var unknown
     */
    public $routeVar = 'r';

    /**
     * 缓存组件名称
     *
     * @var unknown
     */
    public $cache = 'cache';
    public $ruleConfig = 'urlrule1';

    private $_baseUrl;
    private $_hostInfo;


    /**
     * 初始化
     */
    public function init($config = array())
    {
        foreach ( $config as $name => $value ) {
            $this->$name = $value;
        }
        $this->compileRules ();
    }

    /**
     * 解析ControllerPath,并返回解析后的结果集
     *
     * @throws HttpException
     * @return string Ambigous multitype:string multitype: ,
     *         multitype:multitype: string , unknown>
     */
    public function parseRequest()
    {
        $pathInfo = request::getPathInfo ();
        if ( ! $this->enablePrettyUrl || ! $pathInfo || strpos ( $pathInfo, '.php' ) !== false ) { //兼容Get模式URL在Path形式下直接访问

            return false;
        }
        $result = $this->parsePathinfo ( $pathInfo );
        if ( $result !== false ) {
            list ( $route, $params ) = $result;
            $_GET = array_merge ( $_GET, $params );
            return $route;
        }
        return false;
    }

    public function parseUrl(){
        $pathInfo = request::getPathInfo ();
        if ( ! $this->enablePrettyUrl || ! $pathInfo || strpos ( $pathInfo, '.php' ) !== false ) { //兼容Get模式URL在Path形式下直接访问
            $route = request::getQuery ( $this->routeVar );
            if ( is_array ( $route ) ) {
                $route = '';
            }
            return  array (( string ) $route,array () );
        } else {
            return $this->parsePathinfo ( $pathInfo );
        }
    }

    /**
     * Parses the URL rules.
     */
    protected function compileRules()
    {
        if ( ! $this->enablePrettyUrl || empty ( $this->rules ) || strpos ( request::getPathInfo (), '.php' ) !== false ) {
            return;
        }
        //if ( is_string ( $this->cache ) ) {
        //    $this->cache = Leaps::getComponent ( $this->cache );
        //}
        if ( $this->cache instanceof Cache ) {
            $cacheKey = __CLASS__;
            $hash = md5 ( json_encode ( $this->rules ) );
            if ( ($data = $this->cache->get ( __CLASS__ )) !== false && isset ( $data [1] ) && $data [1] === $hash ) {
                $this->rules = $data [0];
                return;
            }
        }

        $rules = array ();
        foreach ( $this->rules as $key => $rule ) {
            if ( ! is_array ( $rule ) ) {
                $rule = array ('route' => $rule );
                if ( preg_match ( '/^((?:(GET|HEAD|POST|PUT|PATCH|DELETE),)*(GET|HEAD|POST|PUT|PATCH|DELETE))\s+(.*)$/', $key, $matches ) ) {
                    $rule ['verb'] = explode ( ',', $matches [1] );
                    $rule ['mode'] = UrlRule::PARSING_ONLY;
                    $key = $matches [4];
                }
                $rule ['pattern'] = $key;
            }
            $ruleobj = new $this->ruleConfig();
            foreach ( $rule as $name => $value ) {
                $ruleobj->$name = $value;
            }
            $rules [] = $ruleobj;
        }
        $this->rules = $rules;
        //if ( isset ( $cacheKey, $hash ) ) {
        //    $this->cache->set ( $cacheKey, array ($this->rules,$hash ) );
        //}
    }

    /**
     * 解析PathInfo
     */
    private function parsePathinfo($rawPathInfo)
    {
        foreach ( $this->rules as $rule ) {
            if ( ($result = $rule->parseRequest ( $this )) !== false ) {
                return $result;
            }
        }
        $pathInfo = $this->removeUrlSuffix ( $rawPathInfo, $this->suffix );
        /**
         * $suffix = ( string ) $this->suffix; if ( $suffix !== '' && $pathInfo
         * !== '' ) { $n = strlen ( $this->suffix ); if ( substr ( $pathInfo, -
         * $n ) === $this->suffix ) { $pathInfo = substr ( $pathInfo, 0, - $n );
         * if ( $pathInfo === '' ) { // suffix alone is not allowed return
         * false; } } else { // suffix doesn't match return false; } }*
         */
        return array ($pathInfo,array () );
    }

    /**
     * 创建URL
     * @param unknown $route
     * @param unknown $params
     * @return string
     */
    public function createUrl($route, $params = array())
    {
        $anchor = isset ( $params ['#'] ) ? '#' . $params ['#'] : '';
        unset ( $params ['#'], $params [$this->routeVar] );
        $route = trim ( $route, '/' );
        $baseUrl = $this->getBaseUrl ();

        if ( $this->enablePrettyUrl ) {
            foreach ( $this->rules as $rule ) {
                if ( ($url = $rule->createUrl ( $this, $route, $params )) !== false ) {
                    if ( $rule->host !== null ) {
                        if ( $baseUrl !== '' && ($pos = strpos ( $url, '/', 8 )) !== false ) {
                            return substr ( $url, 0, $pos ) . $baseUrl . substr ( $url, $pos );
                        } else {
                            return $url . $baseUrl . $anchor;
                        }
                    } else {
                        return "$baseUrl/{$url}{$anchor}";
                    }
                }
            }
            if ( $this->suffix !== null ) {
                $route .= $this->suffix;
            }
            if ( ! empty ( $params ) ) {
                $route .= '?' . http_build_query ( $params );
            }
            return "$baseUrl/{$route}{$anchor}";
        } else {
            $url = "$baseUrl?{$this->routeVar}=$route";
            if ( ! empty ( $params ) ) {
                $url .= '&' . http_build_query ( $params );
            }
            return $url;
        }
    }

    /**
     * 创建绝对地址的URL
     * @param unknown $route
     * @param unknown $params
     * @return string
     */
    public function createAbsoluteUrl($route, $params = array())
    {
        $url = $this->createUrl ( $route, $params );
        if ( strpos ( $url, '://' ) !== false ) {
            return $url;
        } else {
            return $this->getHostInfo () . $url;
        }
    }

    /**
     * Removes the URL suffix from path info.
     *
     * @param string $pathInfo path info part in the URL
     * @param string $urlSuffix the URL suffix to be removed
     * @return string path info with URL suffix removed.
     */
    public function removeUrlSuffix($pathInfo, $urlSuffix)
    {
        if ( $urlSuffix !== '' && substr ( $pathInfo, - strlen ( $urlSuffix ) ) === $urlSuffix )
            return substr ( $pathInfo, 0, - strlen ( $urlSuffix ) );
        else return $pathInfo;
    }
    public function getBaseUrl()
    {
        if ( $this->_baseUrl === null ) {
            $this->_baseUrl = $this->showScriptName || ! $this->enablePrettyUrl ? request::getScriptUrl () : request::getBaseUrl ();
        }
        return $this->_baseUrl;
    }

    /**
     * Sets the base URL that is used by [[createUrl()]] to prepend URLs it
     * creates.
     *
     * @param string $value the base URL that is used by [[createUrl()]] to
     *            prepend URLs it creates.
     */
    public function setBaseUrl($value)
    {
        $this->_baseUrl = rtrim ( $value, '/' );
    }

    /**
     * Returns the host info that is used by [[createAbsoluteUrl()]] to prepend
     * URLs it creates.
     *
     * @return string the host info (e.g. "http://www.example.com") that is used
     *         by [[createAbsoluteUrl()]] to prepend URLs it creates.
     */
    public function getHostInfo()
    {
        if ( $this->_hostInfo === null ) {
            $this->_hostInfo = request::getHostInfo ();
        }
        return $this->_hostInfo;
    }

    /**
     * Sets the host info that is used by [[createAbsoluteUrl()]] to prepend
     * URLs it creates.
     *
     * @param string $value the host info (e.g. "http://www.example.com") that
     *            is used by [[createAbsoluteUrl()]] to prepend URLs it creates.
     */
    public function setHostInfo($value)
    {
        $this->_hostInfo = rtrim ( $value, '/' );
    }
}

/**
 * UrlRule represents a rule used for parsing and generating URLs.
 *
 * @author Qiang Xue <qiang.xue@gmail.com>
 * @since 2.0
 */
class urlrule1
{
    /**
     * Set [[mode]] with this value to mark that this rule is for URL parsing
     * only
     */
    const PARSING_ONLY = 1;
    /**
     * Set [[mode]] with this value to mark that this rule is for URL creation
     * only
     */
    const CREATION_ONLY = 2;

    /**
     *
     * @var string the name of this rule. If not set, it will use [[pattern]] as
     *      the name.
     */
    public $name;
    /**
     *
     * @var string the pattern used to parse and create the path info part of a
     *      URL.
     * @see host
     */
    public $pattern;
    /**
     *
     * @var string the pattern used to parse and create the host info part of a
     *      URL.
     * @see pattern
     */
    public $host;
    /**
     *
     * @var string the route to the controller action
     */
    public $route;
    /**
     *
     * @var array the default GET parameters (name => value) that this rule
     *      provides. When this rule is used to parse the incoming request, the
     *      values declared in this property will be injected into $_GET.
     */
    public $defaults = array ();
    /**
     *
     * @var string the URL suffix used for this rule. For example, ".html" can
     *      be used so that the URL looks like pointing to a static HTML page.
     *      If not, the value of [[UrlManager::suffix]] will be used.
    */
    public $suffix;
    /**
     *
     * @var string array HTTP verb (e.g. GET, POST, DELETE) that this rule
     *      should match. Use array to represent multiple verbs that this rule
     *      may match. If this property is not set, the rule can match any verb.
     *      Note that this property is only used when parsing a request. It is
     *      ignored for URL creation.
     */
    public $verb;
    /**
     *
     * @var integer a value indicating if this rule should be used for both
     *      request parsing and URL creation, parsing only, or creation only. If
     *      not set or 0, it means the rule is both request parsing and URL
     *      creation. If it is [[PARSING_ONLY]], the rule is for request parsing
     *      only. If it is [[CREATION_ONLY]], the rule is for URL creation only.
     */
    public $mode;

    /**
     *
     * @var string the template for generating a new URL. This is derived from
     *      [[pattern]] and is used in generating URL.
     */
    private $_template;
    /**
     *
     * @var string the regex for matching the route part. This is used in
     *      generating URL.
     */
    private $_routeRule;
    /**
     *
     * @var array list of regex for matching parameters. This is used in
     *      generating URL.
     */
    private $_paramRules = array ();
    /**
     *
     * @var array list of parameters used in the route.
    */
    private $_routeParams = array ();

    /**
     * Initializes this rule.
    */
    public function init()
    {
        if ( $this->pattern === null ) {
            throw new Exception ( 'UrlRule::pattern must be set.' );
        }
        if ( $this->route === null ) {
            throw new Exception ( 'UrlRule::route must be set.' );
        }
        if ( $this->verb !== null ) {
            if ( is_array ( $this->verb ) ) {
                foreach ( $this->verb as $i => $verb ) {
                    $this->verb [$i] = strtoupper ( $verb );
                }
            } else {
                $this->verb = array (strtoupper ( $this->verb ) );
            }
        }
        if ( $this->name === null ) {
            $this->name = $this->pattern;
        }

        $this->pattern = trim ( $this->pattern, '/' );

        if ( $this->host !== null ) {
            $this->pattern = rtrim ( $this->host, '/' ) . rtrim ( '/' . $this->pattern, '/' ) . '/';
        } elseif ( $this->pattern === '' ) {
            $this->_template = '';
            $this->pattern = '#^$#u';
            return;
        } else {
            $this->pattern = '/' . $this->pattern . '/';
        }

        $this->route = trim ( $this->route, '/' );
        if ( strpos ( $this->route, '<' ) !== false && preg_match_all ( '/<(\w+)>/', $this->route, $matches ) ) {
            foreach ( $matches [1] as $name ) {
                $this->_routeParams [$name] = "<$name>";
            }
        }

        $tr = $tr2 = array ();
        if ( preg_match_all ( '/<(\w+):?([^>]+)?>/', $this->pattern, $matches, PREG_OFFSET_CAPTURE | PREG_SET_ORDER ) ) {
            foreach ( $matches as $match ) {
                $name = $match [1] [0];
                $pattern = isset ( $match [2] [0] ) ? $match [2] [0] : '[^\/]+';
                if ( isset ( $this->defaults [$name] ) ) {
                    $length = strlen ( $match [0] [0] );
                    $offset = $match [0] [1];
                    if ( $offset > 1 && $this->pattern [$offset - 1] === '/' && $this->pattern [$offset + $length] === '/' ) {
                        $tr ["/<$name>"] = "(/(?P<$name>$pattern))?";
                    } else {
                        $tr ["<$name>"] = "(?P<$name>$pattern)?";
                    }
                } else {
                    $tr ["<$name>"] = "(?P<$name>$pattern)";
                }
                if ( isset ( $this->_routeParams [$name] ) ) {
                    $tr2 ["<$name>"] = "(?P<$name>$pattern)";
                } else {
                    $this->_paramRules [$name] = $pattern === '[^\/]+' ? '' : "#^$pattern$#";
                }
            }
        }
        $tr ['.'] = '\\.';

        $this->_template = preg_replace ( '/<(\w+):?([^>]+)?>/', '<$1>', $this->pattern );
        $this->pattern = '#^' . trim ( strtr ( $this->_template, $tr ), '/' ) . '$#u';

        if ( ! empty ( $this->_routeParams ) ) {
            $this->_routeRule = '#^' . strtr ( $this->route, $tr2 ) . '$#u';
        }
    }

    /**
     * Parses the given request and returns the corresponding route and
     * parameters.
     *
     * @param UrlManager $manager the URL manager
     * @param Request $request the request component
     * @return array boolean parsing result. The route and the parameters are
     *         returned as an array. If false, it means this rule cannot be used
     *         to parse this path info.
     */
    public function parseRequest($manager)
    {
        if ( $this->mode === self::CREATION_ONLY ) {
            return false;
        }

        if ( $this->verb !== null && ! in_array ( request::getRequestType (), $this->verb, true ) ) {
            return false;
        }

        $pathInfo = request::getPathInfo();
        $suffix = (string)($this->suffix === null ? $manager->suffix : $this->suffix);
        if ($suffix !== '' && $pathInfo !== '') {
            $manager->removeUrlSuffix($pathInfo, $suffix);
            /**
             $n = strlen($suffix);
             if (substr($pathInfo, -$n) === $suffix) {
             $pathInfo = substr($pathInfo, 0, -$n);
             if ($pathInfo === '') {
             // suffix alone is not allowed
             return false;
             }
             } else {
             return false;
             }
            **/
        }

        if ( $this->host !== null ) {
            $pathInfo = strtolower ( request::getHostInfo () ) . '/' . $pathInfo;
        }

        if ( ! preg_match ( $this->pattern, $pathInfo, $matches ) ) {
            return false;
        }
        foreach ( $this->defaults as $name => $value ) {
            if ( ! isset ( $matches [$name] ) || $matches [$name] === '' ) {
                $matches [$name] = $value;
            }
        }
        $params = $this->defaults;
        $tr = array ();
        foreach ( $matches as $name => $value ) {
            if ( isset ( $this->_routeParams [$name] ) ) {
                $tr [$this->_routeParams [$name]] = $value;
                unset ( $params [$name] );
            } elseif ( isset ( $this->_paramRules [$name] ) ) {
                $params [$name] = $value;
            }
        }
        if ( $this->_routeRule !== null ) {
            $route = strtr ( $this->route, $tr );
        } else {
            $route = $this->route;
        }
        return array ($route,$params );
    }

    /**
     * Creates a URL according to the given route and parameters.
     *
     * @param UrlManager $manager the URL manager
     * @param string $route the route. It should not have slashes at the
     *            beginning or the end.
     * @param array $params the parameters
     * @return string boolean created URL, or false if this rule cannot be used
     *         for creating this URL.
     */
    public function createUrl($manager, $route, $params)
    {
        if ( $this->mode === self::PARSING_ONLY ) {
            return false;
        }

        $tr = array ();

        // match the route part first
        if ( $route !== $this->route ) {
            if ( $this->_routeRule !== null && preg_match ( $this->_routeRule, $route, $matches ) ) {
                foreach ( $this->_routeParams as $name => $token ) {
                    if ( isset ( $this->defaults [$name] ) && strcmp ( $this->defaults [$name], $matches [$name] ) === 0 ) {
                        $tr [$token] = '';
                    } else {
                        $tr [$token] = $matches [$name];
                    }
                }
            } else {
                return false;
            }
        }

        // match default params
        // if a default param is not in the route pattern, its value must also be matched
        foreach ( $this->defaults as $name => $value ) {
            if ( isset ( $this->_routeParams [$name] ) ) {
                continue;
            }
            if ( ! isset ( $params [$name] ) ) {
                return false;
            } elseif ( strcmp ( $params [$name], $value ) === 0 ) { // strcmp will do string conversion automatically
                unset ( $params [$name] );
                if ( isset ( $this->_paramRules [$name] ) ) {
                    $tr ["<$name>"] = '';
                }
            } elseif ( ! isset ( $this->_paramRules [$name] ) ) {
                return false;
            }
        }

        // match params in the pattern
        foreach ( $this->_paramRules as $name => $rule ) {
            if ( isset ( $params [$name] ) && ($rule === '' || preg_match ( $rule, $params [$name] )) ) {
                $tr ["<$name>"] = urlencode ( $params [$name] );
                unset ( $params [$name] );
            } elseif ( ! isset ( $this->defaults [$name] ) || isset ( $params [$name] ) ) {
                return false;
            }
        }

        $url = trim ( strtr ( $this->_template, $tr ), '/' );
        if ( $this->host !== null ) {
            $pos = strpos ( $url, '/', 8 );
            if ( $pos !== false ) {
                $url = substr ( $url, 0, $pos ) . preg_replace ( '#/+#', '/', substr ( $url, $pos ) );
            }
        } elseif ( strpos ( $url, '//' ) !== false ) {
            $url = preg_replace ( '#/+#', '/', $url );
        }

        if ( $url !== '' ) {
            $url .= ($this->suffix === null ? $manager->suffix : $this->suffix);
        }

        if ( ! empty ( $params ) ) {
            $url .= '?' . http_build_query ( $params );
        }
        return $url;
    }
}
